/*
 * approx_esc_1d.cpp
 *
 *  Created on: Jul 30, 2012
 *      Author: Berk Calli
 *      Organization: Delft Biorobotics Lab., Delft University of Technology
 *		Contact info: b.calli@tudelft.nl, web: www.dbl.tudelft.nl
 *
 * Class for one dimensional approximation based extremum seeking control
 *
 * * References:
 * - C. Zhang and R. Ordonez, “Robust and adaptive design of numerical optimization-based extremum seeking control,” Automatica, vol. 45, pp. 634–646, 2009.
 * - B. Calli, W. Caarls, P. Jonker and M. Wisse, "Comparison of Extremum Seeking Control Algorithms for Robotic Applications," IROS 2012.
 */

#include "esc_approx/approx_esc_1d.h"

ApproxESC1D::ApproxESC1D(){
	data_size_ = 0;
	poly_degree_ = 0;
	k_grad_ = 0;
	init_vel_ = 0;
	sampling_ = 0;
	ptr_ = 0;
	initialized_ = false;
}

ApproxESC1D::ApproxESC1D(int data_size,int poly_degree, double k_grad, double init_vel, int sampling){
	init(data_size,poly_degree,k_grad,init_vel,sampling);
}

void ApproxESC1D::init(int data_size,int poly_degree, double k_grad, double init_vel, int sampling){
	data_size_ = data_size;
	poly_degree_ = poly_degree;
	k_grad_ = k_grad;
	init_vel_ = init_vel;
	sampling_ = sampling;
	states_.resize(data_size);
	obj_vals_.resize(data_size);
	sample_ = 0;
	ptr_ = 0;
	initialized_ = true;
}

std::vector<double> ApproxESC1D::step(std::vector<double> state, double obj_val){
	if(initialized_){
		if(sample_ % sampling_ == 0){

			states_(ptr_) = state[0];
			obj_vals_(ptr_) = obj_val;

			Eigen::MatrixXf V(data_size_,poly_degree_+1);
			V = Eigen::MatrixXf::Zero(data_size_, poly_degree_+1);

			for (int i = 0; i<data_size_; i++)
				for (int j = 0; j<poly_degree_+1; j++)
					V(i,j) = std::pow(states_(i),(poly_degree_-j));

			Eigen::VectorXf coef(data_size_);

			coef = V.colPivHouseholderQr().solve(obj_vals_);
			state_curr_ = states_(ptr_);
			double grad_val = 0;
			Eigen::VectorXf vec(data_size_);

			for (int i = 0; i<poly_degree_; i++)
				grad_val = grad_val + coef(i)*(poly_degree_-i)*std::pow(state_curr_,(poly_degree_-i-1));

			if (sample_<sampling_*data_size_+1){
				vel_ref_ = init_vel_;
			}
			else{
				vel_ref_ = -k_grad_*grad_val;
				if (vel_ref_!= vel_ref_)
					vel_ref_ = 0;
			}

			sample_ = sample_+1;
			ptr_ = ptr_+1;
			if(ptr_>=data_size_)
				ptr_ = 0;

			std::vector<double> out;
			out.push_back(vel_ref_);
			return out;
		}
		else{
			sample_ = sample_+1;
			std::vector<double> out;
			out.push_back(vel_ref_);
			return out;
		}
	}
	else{
		fprintf(stderr,"The approximation based ESC (1D) is not initialized... It will not be executed. \n");
		return std::vector<double>();
	}
}

ESC::inputType ApproxESC1D::getInputType(){
	return ESC::inputStateValue;
}
ESC::outputType ApproxESC1D::getOutputType(){
	return ESC::outputVelocity;
}

std::vector<double> ApproxESC1D::monitor(){
	return std::vector<double> ();
}
std::vector<std::string> ApproxESC1D::monitorNames(){
	return std::vector<std::string>();
}

void ApproxESC1D::reset(){
	sample_ = 0;
	ptr_ = 0;
}
